<?php
/**
 * La constante para nuestro hook de cron personalizado.
 */
define('AICS_CRON_HOOK', 'aics_synthesis_cron_hook');

// Cargar las clases de la lógica del plugin
require_once plugin_dir_path( __FILE__ ) . 'class-aics-logger.php';
require_once plugin_dir_path( __FILE__ ) . 'trait-aics-api-style-helper.php';
require_once plugin_dir_path( __FILE__ ) . 'class-aics-gemini-api.php';
require_once plugin_dir_path( __FILE__ ) . 'class-aics-openai-api.php';
require_once plugin_dir_path( __FILE__ ) . 'class-aics-engine.php';

/**
 * La clase principal del plugin.
 */
class AICS_Core {
	private $plugin_name;
	private $version;
	private $options;
	private $dashboard_page_hook;
	private $settings_page_hook;

	public function __construct() {
		$this->plugin_name = 'ai-content-synthesizer';
		$this->version = '1.0.1';
		$this->options = get_option('aics_settings');
	}

	public function run() {
		add_action( 'admin_menu', array( $this, 'add_plugin_admin_menu' ) );
		add_action( 'admin_init', array( $this, 'register_plugin_settings' ) );
		add_action( 'admin_init', array( $this, 'handle_source_actions' ) );
		add_action( 'admin_init', array( $this, 'handle_dashboard_actions' ) );
		add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_admin_styles_and_scripts' ) );
		add_action( AICS_CRON_HOOK, array( $this, 'run_engine_cycle' ) );

		// NUEVO: Registrar las acciones AJAX
		add_action( 'wp_ajax_aics_create_draft_from_modal', array( $this, 'ajax_create_draft_from_modal' ) );
		add_action( 'wp_ajax_aics_publish_directly_from_modal', array( $this, 'ajax_publish_directly_from_modal' ) );
		add_action( 'wp_ajax_aics_schedule_post_from_modal', array( $this, 'ajax_schedule_post_from_modal' ) );
		add_action( 'wp_ajax_aics_save_modal_changes', array( $this, 'ajax_save_modal_changes' ) );
		add_action( 'wp_ajax_aics_regenerate_content', array( $this, 'ajax_regenerate_content' ) );
		add_action( 'wp_ajax_aics_load_rss_feeds', array( $this, 'ajax_load_rss_feeds' ) );
		add_action( 'wp_ajax_aics_force_load_feeds', array( $this, 'ajax_force_load_feeds' ) );

		// NUEVO: Añadir enlaces personalizados en la página de plugins
		add_filter( 'plugin_row_meta', array( $this, 'add_plugin_meta_links' ), 10, 2 );
	}

    public function run_engine_cycle() {
        // Ahora, el ciclo del motor solo carga los feeds en el dashboard.
        $new_articles_count = $this->_load_rss_feeds_logic();
        $message = sprintf(
            _n(
                'El proceso automático ha encontrado %d nuevo artículo.',
                'El proceso automático ha encontrado %d nuevos artículos.',
                $new_articles_count,
                'ai-content-synthesizer'
            ), $new_articles_count);
        AICS_Logger::log('INFO', $message);
    }

	public function enqueue_admin_styles_and_scripts( $hook ) {
		if ( $hook != $this->dashboard_page_hook && $hook != $this->settings_page_hook ) {
			return;
		}
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . '../admin/css/aics-admin-styles.css', array(), $this->version, 'all' );
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . '../admin/js/aics-admin-scripts.js', array('jquery'), $this->version, true );

		// NUEVO: Pasar datos de PHP a JavaScript (como el nonce para AJAX)
		wp_localize_script( $this->plugin_name, 'aics_ajax_object', [
			'ajax_url' => admin_url( 'admin-ajax.php' ),
			'nonce'    => wp_create_nonce( 'aics_ajax_nonce' )
		]);
	}

	public function register_plugin_settings() {
		register_setting( 'aics_settings_group', 'aics_settings', array( $this, 'sanitize_general_settings' ) );
		add_settings_section( 'aics_api_section', 'Configuración de la IA', null, 'aics_settings_tab' );
		add_settings_field( 'aics_ai_provider', 'Proveedor de IA', array( $this, 'render_ai_provider_field' ), 'aics_settings_tab', 'aics_api_section' );
		add_settings_field( 'aics_gemini_api_key', 'Clave API de Google Gemini', array( $this, 'render_gemini_key_field' ), 'aics_settings_tab', 'aics_api_section' );
		add_settings_field( 'aics_gemini_model', 'Modelo de Gemini', array( $this, 'render_gemini_model_field' ), 'aics_settings_tab', 'aics_api_section' );
		add_settings_field( 'aics_openai_api_key', 'Clave API de OpenAI', array( $this, 'render_openai_key_field' ), 'aics_settings_tab', 'aics_api_section' );
        add_settings_section( 'aics_ia_params_section', 'Parámetros de Generación de la IA', null, 'aics_settings_tab' );
        add_settings_field( 'aics_writing_style', 'Estilo de Redacción', array( $this, 'render_writing_style_field' ), 'aics_settings_tab', 'aics_ia_params_section' );
        add_settings_field( 'aics_article_length', 'Longitud del Artículo', array( $this, 'render_article_length_field' ), 'aics_settings_tab', 'aics_ia_params_section' );
        add_settings_field( 'aics_enable_full_content_scraping', 'Scraping de Contenido Completo', array( $this, 'render_enable_full_content_scraping_field' ), 'aics_settings_tab', 'aics_ia_params_section' );
        add_settings_field( 'aics_max_prompt_chars', 'Máx. Caracteres para IA', array( $this, 'render_max_prompt_chars_field' ), 'aics_settings_tab', 'aics_ia_params_section' );
		add_settings_section( 'aics_generation_section', 'Reglas de Ejecución', null, 'aics_settings_tab' );
		add_settings_field( 'aics_crawl_frequency', 'Frecuencia de rastreo', array( $this, 'render_crawl_frequency_field' ), 'aics_settings_tab', 'aics_generation_section' );
	}

	public function sanitize_general_settings( $input ) {
		$new_input = $this->options;
		if ( isset( $input['ai_provider'] ) ) { $new_input['ai_provider'] = sanitize_text_field( $input['ai_provider'] ); }
		if ( isset( $input['gemini_api_key'] ) ) { $new_input['gemini_api_key'] = sanitize_text_field( $input['gemini_api_key'] ); }
		if ( isset( $input['gemini_model'] ) ) { $new_input['gemini_model'] = sanitize_text_field( $input['gemini_model'] ); }
		if ( isset( $input['openai_api_key'] ) ) { $new_input['openai_api_key'] = sanitize_text_field( $input['openai_api_key'] ); }
		if ( isset( $input['writing_style'] ) ) { $new_input['writing_style'] = sanitize_text_field( $input['writing_style'] ); }
		$new_input['enable_full_content_scraping'] = isset( $input['enable_full_content_scraping'] ) ? 1 : 0;
		if ( isset( $input['max_prompt_chars'] ) ) { $new_input['max_prompt_chars'] = absint( $input['max_prompt_chars'] ); }
		if ( isset( $input['article_length'] ) ) { $new_input['article_length'] = sanitize_text_field( $input['article_length'] ); }
		if ( isset( $input['crawl_frequency'] ) ) { $new_input['crawl_frequency'] = sanitize_text_field( $input['crawl_frequency'] ); }
		if ( isset( $input['rss_feeds'] ) && is_array( $input['rss_feeds'] ) ) {
			$new_input['rss_feeds'] = $input['rss_feeds'];
		}
		return $new_input;
	}

	public function render_ai_provider_field() { $provider = $this->options['ai_provider'] ?? 'gemini'; ?> <select id="aics_ai_provider" name="aics_settings[ai_provider]"> <option value="gemini" <?php selected($provider, 'gemini'); ?>>Google Gemini</option> <option value="openai" <?php selected($provider, 'openai'); ?>>OpenAI (ChatGPT)</option> </select> <p class="description">Selecciona el servicio de IA que quieres utilizar.</p> <?php }
	public function render_gemini_key_field() { $key = $this->options['gemini_api_key'] ?? ''; ?> <div class="api-key-field-wrapper" data-provider="gemini"> <input type="password" name="aics_settings[gemini_api_key]" value="<?php echo esc_attr($key); ?>" class="regular-text"> </div> <?php }
	public function render_gemini_model_field() { 
		$current_model = $this->options['gemini_model'] ?? 'gemini-2.5-flash'; 
		$models = [
			'gemini-2.5-flash' => 'Gemini 2.5 Flash (Rápido, recomendado)',
			'gemini-2.5-pro'   => 'Gemini 2.5 Pro (Potente)',
		];
		?>
		<div class="api-key-field-wrapper" data-provider="gemini">
			<select name="aics_settings[gemini_model]">
				<?php foreach ($models as $key => $label): ?>
					<option value="<?php echo esc_attr($key); ?>" <?php selected($current_model, $key); ?>><?php echo esc_html($label); ?></option>
				<?php endforeach; ?>
			</select>
			<p class="description">Selecciona el modelo de Gemini a utilizar. Flash es más rápido y económico.</p>
		</div>
		<?php 
	}
	public function render_enable_full_content_scraping_field() {
		$checked = isset($this->options['enable_full_content_scraping']) && $this->options['enable_full_content_scraping'] == 1;
		?>
		<label><input type="checkbox" name="aics_settings[enable_full_content_scraping]" value="1" <?php checked(1, $checked); ?>> <span class="description">Si se marca, el plugin intentará extraer el contenido completo de la URL original del artículo en lugar de usar solo el resumen del feed RSS.</span></label>
		<?php
	}
	public function render_max_prompt_chars_field() {
		$max_chars = isset( $this->options['max_prompt_chars'] ) ? $this->options['max_prompt_chars'] : 3000; // Valor por defecto
		printf( '<input type="number" name="aics_settings[max_prompt_chars]" value="%d" min="500" max="10000" class="small-text">', esc_attr( $max_chars ) ); echo '<p class="description">Número máximo de caracteres del contenido de la fuente que se enviará a la IA. Un valor más alto permite a la IA generar artículos más detallados, pero puede aumentar el consumo de tokens.</p>';
	}
	public function render_article_length_field() { $current_length = $this->options['article_length'] ?? 'medium'; $lengths = [ 'short' => 'Corto (~200 palabras)', 'medium' => 'Medio (~400 palabras)', 'long' => 'Largo (~600+ palabras)' ]; ?> <select name="aics_settings[article_length]"> <?php foreach ($lengths as $key => $label): ?> <option value="<?php echo esc_attr($key); ?>" <?php selected($current_length, $key); ?>> <?php echo esc_html($label); ?> </option> <?php endforeach; ?> </select> <p class="description">Define la longitud aproximada de los artículos que generará la IA.</p> <?php }
	public function render_openai_key_field() { $key = $this->options['openai_api_key'] ?? ''; ?> <div class="api-key-field-wrapper" data-provider="openai"> <input type="password" name="aics_settings[openai_api_key]" value="<?php echo esc_attr($key); ?>" class="regular-text"> </div> <?php }
	public function render_writing_style_field() { $current_style = $this->options['writing_style'] ?? 'formal'; $styles = [ 'formal' => 'Formal y Periodístico', 'informal' => 'Informal y Conversacional', 'sensacionalista' => 'Sensacionalista (Clickbait)', 'profesional' => 'Profesional y Corporativo', 'creativo' => 'Creativo y Entusiasta', 'tecnico' => 'Técnico y Detallado' ]; ?> <select name="aics_settings[writing_style]"> <?php foreach ($styles as $key => $label): ?> <option value="<?php echo esc_attr($key); ?>" <?php selected($current_style, $key); ?>> <?php echo esc_html($label); ?> </option> <?php endforeach; ?> </select> <p class="description">Define el tono y el estilo que la IA debe utilizar al redactar los artículos.</p> <?php }
	public function render_crawl_frequency_field() { 
		$frequency = isset( $this->options['crawl_frequency'] ) ? $this->options['crawl_frequency'] : 'hourly'; 
		$schedules = wp_get_schedules();
		
		echo '<div class="aics-cron-trigger-wrapper">'; // NUEVO: Contenedor Flex
		echo '<select name="aics_settings[crawl_frequency]">'; 
		foreach($schedules as $key => $value) { 
			printf( '<option value="%s" %s>%s</option>', esc_attr($key), selected($frequency, $key, false), esc_html($value['display']) ); 
		} 
		echo '</select>';
		// NUEVO: Botón para forzar la ejecución
		submit_button('Forzar Ejecución Ahora', 'secondary', 'aics_force_cron', false);
		// NUEVO: Indicador de carga que se mostrará con JS
		echo '<span id="aics-processing-spinner" style="display:none; vertical-align: middle; margin-left: 10px;"><span class="spinner is-active" style="margin-right: 5px;"></span> Procesando...</span>';
		echo '</div>'; // Fin del contenedor
		echo '<p class="description">Con qué frecuencia el sistema cargará nuevos artículos de los feeds en el dashboard. Puedes forzar una carga manual con el botón.</p>'; 
	}
	public function add_plugin_admin_menu() { $this->dashboard_page_hook = add_menu_page('AI Content Synthesizer', 'AI Synthesizer', 'manage_options', $this->plugin_name, array( $this, 'display_dashboard_page' ), 'dashicons-analytics', 25 ); 
		
		add_submenu_page($this->plugin_name, 'Dashboard', 'Dashboard', 'manage_options', $this->plugin_name, array( $this, 'display_dashboard_page' ) ); 
		
		$this->settings_page_hook = add_submenu_page($this->plugin_name, 'Ajustes', 'Ajustes', 'manage_options', $this->plugin_name . '-settings', array( $this, 'display_settings_page' ) ); 
	}
	public function display_dashboard_page() { require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/aics-admin-tab-dashboard.php'; } // Se ha eliminado el logo de aquí
	public function display_settings_page() { require_once plugin_dir_path( dirname( __FILE__ ) ) . 'admin/partials/aics-admin-settings-page.php'; } // Se ha eliminado el logo de aquí
	
	public function handle_source_actions() {
			// Variable para saber si debemos redirigir al final
			$action_taken = false;

			// --- PROCESAR ACCIONES POST (Añadir / Actualizar) ---
			if ( isset( $_POST['aics_source_nonce'] ) && wp_verify_nonce( $_POST['aics_source_nonce'], 'aics_source_action' ) ) {
				
				$options = get_option('aics_settings', []);
				if ( !isset($options['rss_feeds']) || !is_array($options['rss_feeds']) ) {
					$options['rss_feeds'] = array();
				}

				// Acción: Añadir nueva fuente
				if ( isset( $_POST['action'] ) && $_POST['action'] === 'aics_add_source' ) {
					$name = sanitize_text_field( $_POST['source_name'] );
					$url  = esc_url_raw( $_POST['source_url'] );

					if ( ! empty( $name ) && filter_var( $url, FILTER_VALIDATE_URL ) ) {
						$source_id = uniqid('source_');
						$options['rss_feeds'][$source_id] = [ 'name' => $name, 'url'  => $url ];
						$action_taken = true; // <-- CORRECCIÓN: Actualizamos la variable
					}
				}

				// Acción: Actualizar fuente existente
				if ( isset( $_POST['action'] ) && $_POST['action'] === 'aics_update_source' ) {
					$id   = sanitize_key( $_POST['source_id'] );
					$name = sanitize_text_field( $_POST['source_name'] );
					$url  = esc_url_raw( $_POST['source_url'] );

					if ( isset( $options['rss_feeds'][$id] ) && ! empty( $name ) && filter_var( $url, FILTER_VALIDATE_URL ) ) {
						$options['rss_feeds'][$id]['name'] = $name;
						$options['rss_feeds'][$id]['url'] = $url;
						$action_taken = true; // <-- CORRECCIÓN: Actualizamos la variable
					}
				}
				
				// Si se realizó una acción POST válida, guardar en la BD
				if ($action_taken) {
					update_option( 'aics_settings', $options );
				}
			}
			
			// --- PROCESAR ACCIONES GET (Eliminar) ---
			if ( isset( $_GET['action'], $_GET['source_id'], $_GET['_wpnonce'] ) && $_GET['action'] === 'delete' ) {
				if ( wp_verify_nonce( $_GET['_wpnonce'], 'aics_delete_source_' . $_GET['source_id'] ) ) {
					$options = get_option('aics_settings', []);
					$id_to_delete = sanitize_key( $_GET['source_id'] );

					if ( isset( $options['rss_feeds'][$id_to_delete] ) ) {
						unset( $options['rss_feeds'][$id_to_delete] );
						update_option( 'aics_settings', $options );
						$action_taken = true; // <-- CORRECCIÓN: Actualizamos la variable
					}
				}
			}

			// --- REDIRECCIONAR SI SE REALIZÓ ALGUNA ACCIÓN ---
			if ( $action_taken ) {
				$settings_page_url = menu_page_url( $this->plugin_name . '-settings', false );
				wp_safe_redirect( add_query_arg( 'tab', 'sources', $settings_page_url ) );
				exit;
			}
		}

	/**
	 * NUEVO: Añade enlaces personalizados (como "Ajustes") en la fila del plugin en la página de plugins.
	 *
	 * @param array  $links Los enlaces existentes.
	 * @param string $file  El fichero del plugin.
	 * @return array Los enlaces modificados.
	 */
	public function add_plugin_meta_links( $links, $file ) {
		// Construimos el nombre base del plugin para asegurarnos de que solo modificamos nuestro plugin.
		$plugin_basename = plugin_basename( dirname( plugin_dir_path( __FILE__ ) ) . '/' . $this->plugin_name . '.php' );

		if ( $file == $plugin_basename ) {
			// Creamos un nuevo enlace a la página de ajustes del plugin.
			$settings_link = '<a href="' . menu_page_url( $this->plugin_name . '-settings', false ) . '">' . __( 'Ajustes', 'ai-content-synthesizer' ) . '</a>';
			
			// Añadimos el nuevo enlace al array de enlaces existentes.
			array_unshift( $links, $settings_link );
		}

		return $links;
	}

	public function handle_dashboard_actions() {
		// Comprobar si se está enviando una acción en lote desde el dashboard.
		if ( isset( $_POST['action'] ) && $_POST['action'] !== '-1' && isset( $_POST['aics_dashboard_nonce'] ) ) {
			
			if ( ! wp_verify_nonce( $_POST['aics_dashboard_nonce'], 'aics_bulk_action_dashboard' ) ) {
				wp_die( 'La verificación de seguridad ha fallado.' );
			}

			if ( empty( $_POST['article_ids'] ) || ! is_array( $_POST['article_ids'] ) ) {
				// No se seleccionó ningún artículo, no hacer nada.
				return;
			}

			$action = sanitize_key( $_POST['action'] );
			$article_ids = array_map( 'absint', $_POST['article_ids'] );

			global $wpdb;
			$table_name = $wpdb->prefix . 'aics_synthesized_articles';

			foreach ( $article_ids as $article_id ) {
				if ( $action === 'bulk_create_draft' ) {
					$article = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d AND status = 'pending_review'", $article_id ) );
					if ( $article ) {
						$post_data = [
							'post_title'   => wp_strip_all_tags( $article->title ), 
							'post_content' => $article->content, 
							'post_status'  => 'draft', 
							'post_author'  => get_current_user_id(),
							'tags_input'   => $article->suggested_tags ?? ''
						];
						$post_id = wp_insert_post( $post_data, true );
						if ( ! is_wp_error( $post_id ) ) {
							$wpdb->update( $table_name, [ 'status' => 'published' ], [ 'id' => $article_id ] );
						}
					}
				} elseif ( $action === 'bulk_delete' ) {
					$wpdb->delete( $table_name, [ 'id' => $article_id ], [ '%d' ] );
				}
			}

			// Redirigir para refrescar la página y evitar reenvíos del formulario.
			wp_safe_redirect( menu_page_url( $this->plugin_name, false ) );
			exit;
		}

		// Comprobar si se está enviando una acción individual (desde los botones de cada fila).
		if ( isset( $_GET['action'] ) && in_array( $_GET['action'], ['create_draft', 'delete'] ) && isset( $_GET['article_id'] ) ) {
			$article_id = absint( $_GET['article_id'] );
			$action = sanitize_key( $_GET['action'] );

			if ( ! isset( $_GET['_wpnonce'] ) || ! wp_verify_nonce( $_GET['_wpnonce'], 'aics_single_action_dashboard_' . $article_id ) ) {
				wp_die( 'La verificación de seguridad ha fallado.' );
			}

			global $wpdb;
			$table_name = $wpdb->prefix . 'aics_synthesized_articles';

			if ( $action === 'create_draft' ) {
				$article = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d AND status = 'pending_review'", $article_id ) );
				if ( $article ) {
					$post_data = [
						'post_title'   => wp_strip_all_tags( $article->title ), 
						'post_content' => $article->content, 
						'post_status'  => 'draft', 
						'post_author'  => get_current_user_id(),
						'tags_input'   => $article->suggested_tags ?? ''
					];
					$post_id = wp_insert_post( $post_data, true );
					if ( ! is_wp_error( $post_id ) ) {
						$wpdb->update( $table_name, [ 'status' => 'published' ], [ 'id' => $article_id ] );
					}
				}
			} elseif ( $action === 'delete' ) {
				$wpdb->delete( $table_name, [ 'id' => $article_id ], [ '%d' ] );
			}

			// Redirigir para refrescar la página.
			wp_safe_redirect( menu_page_url( $this->plugin_name, false ) );
			exit;
		}
	}

	/**
	 * NUEVO: Manejador AJAX para crear un borrador desde la modal.
	 */
	public function ajax_create_draft_from_modal() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		// 2. Validar ID
		$article_id = isset( $_POST['article_id'] ) ? absint( $_POST['article_id'] ) : 0;
		if ( empty( $article_id ) ) {
			wp_send_json_error( [ 'message' => 'Falta el ID del artículo.' ], 400 );
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'aics_synthesized_articles';

		// 3. Obtener los datos más recientes del artículo de nuestra tabla
		$article = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d AND status = 'pending_review'", $article_id ) );

		if ( ! $article ) {
			wp_send_json_error( [ 'message' => 'El artículo no se encontró o ya fue procesado.' ], 404 );
		}

		// 4. Crear el post en borrador
		$post_data = [
			'post_title'   => $article->title, 
			'post_content' => $article->content, 
			'post_status'  => 'draft', 
			'post_author'  => get_current_user_id(),
			'tags_input'   => $article->suggested_tags ?? ''
		];
		$post_id = wp_insert_post( $post_data, true );

		if ( is_wp_error( $post_id ) ) {
			wp_send_json_error( [ 'message' => 'Error al crear el borrador: ' . $post_id->get_error_message() ], 500 );
		}

		// 5. Marcar como publicado en nuestra tabla
		$log_message = sprintf('Artículo "%s" (ID: %d) enviado como borrador.', $article->title, $article_id);
		AICS_Logger::log('INFO', $log_message, null, $article_id);

		$wpdb->update( $table_name, [ 'status' => 'published' ], [ 'id' => $article_id ] );
		wp_send_json_success( [ 'message' => 'Borrador creado con éxito.', 'post_id' => $post_id ] );
	}

	/**
	 * NUEVO: Manejador AJAX para publicar un post directamente desde la modal.
	 */
	public function ajax_publish_directly_from_modal() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		// 2. Validar ID
		$article_id = isset( $_POST['article_id'] ) ? absint( $_POST['article_id'] ) : 0;
		if ( empty( $article_id ) ) {
			wp_send_json_error( [ 'message' => 'Falta el ID del artículo.' ], 400 );
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'aics_synthesized_articles';

		// 3. Obtener los datos del artículo
		$article = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d AND status = 'pending_review'", $article_id ) );
		if ( ! $article ) {
			wp_send_json_error( [ 'message' => 'El artículo no se encontró o ya fue procesado.' ], 404 );
		}

		// 4. Crear el post con estado 'publish'
		$post_data = [
			'post_title'   => $article->title, 
			'post_content' => $article->content, 
			'post_status'  => 'publish', // La única diferencia clave
			'post_author'  => get_current_user_id(),
			'tags_input'   => $article->suggested_tags ?? ''
		];
		$post_id = wp_insert_post( $post_data, true );

		if ( is_wp_error( $post_id ) ) {
			wp_send_json_error( [ 'message' => 'Error al publicar el post: ' . $post_id->get_error_message() ], 500 );
		}

		$log_message = sprintf('Artículo "%s" (ID: %d) publicado directamente.', $article->title, $article_id);
		AICS_Logger::log('INFO', $log_message, null, $article_id);

		$wpdb->update( $table_name, [ 'status' => 'published' ], [ 'id' => $article_id ] );
		wp_send_json_success( [ 'message' => 'Post publicado con éxito.', 'post_id' => $post_id ] );
	}

	/**
	 * NUEVO: Manejador AJAX para programar un post desde la modal.
	 */
	public function ajax_schedule_post_from_modal() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		// 2. Validar ID y fecha/hora
		$article_id = isset( $_POST['article_id'] ) ? absint( $_POST['article_id'] ) : 0;
		$schedule_datetime = isset( $_POST['schedule_datetime'] ) ? sanitize_text_field( $_POST['schedule_datetime'] ) : '';

		if ( empty( $article_id ) || empty( $schedule_datetime ) ) {
			wp_send_json_error( [ 'message' => 'Faltan datos requeridos (ID o fecha/hora).' ], 400 );
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'aics_synthesized_articles';

		// 3. Obtener los datos del artículo
		$article = $wpdb->get_row( $wpdb->prepare( "SELECT * FROM $table_name WHERE id = %d AND status = 'pending_review'", $article_id ) );
		if ( ! $article ) {
			wp_send_json_error( [ 'message' => 'El artículo no se encontró o ya fue procesado.' ], 404 );
		}

		// 4. Crear el post con estado 'future' y las fechas correspondientes
		$post_data = [
			'post_title'   => $article->title, 
			'post_content' => $article->content, 
			'post_status'  => 'future', // Estado para posts programados
			'post_author'  => get_current_user_id(),
			'tags_input'   => $article->suggested_tags ?? '',
			'post_date'    => get_date_from_gmt( $schedule_datetime ), // Fecha en la zona horaria del sitio
			'post_date_gmt'=> $schedule_datetime, // Fecha en GMT
		];
		$post_id = wp_insert_post( $post_data, true );

		if ( is_wp_error( $post_id ) ) {
			wp_send_json_error( [ 'message' => 'Error al programar el post: ' . $post_id->get_error_message() ], 500 );
		}

		$log_message = sprintf('Artículo "%s" (ID: %d) programado para: %s.', $article->title, $article_id, $schedule_datetime);
		AICS_Logger::log('INFO', $log_message, null, $article_id);

		$wpdb->update( $table_name, [ 'status' => 'published' ], [ 'id' => $article_id ] );
		wp_send_json_success( [ 'message' => 'Post programado con éxito.', 'post_id' => $post_id ] );
	}

	/**
	 * NUEVO: Manejador AJAX para guardar los cambios de la modal sin crear un borrador.
	 */
	public function ajax_save_modal_changes() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		// 2. Validar datos
		$article_id = isset( $_POST['article_id'] ) ? absint( $_POST['article_id'] ) : 0;
		$new_title = isset( $_POST['new_title'] ) ? sanitize_text_field( $_POST['new_title'] ) : '';
		$new_tags = isset( $_POST['new_tags'] ) ? sanitize_text_field( $_POST['new_tags'] ) : '';

		if ( empty( $article_id ) || empty( $new_title ) ) {
			wp_send_json_error( [ 'message' => 'Faltan datos (ID o título).' ], 400 );
		}

		// 3. Actualizar la base de datos
		global $wpdb;
		$table_name = $wpdb->prefix . 'aics_synthesized_articles';
		$updated = $wpdb->update( $table_name, [ 'title' => $new_title, 'suggested_tags' => $new_tags ], [ 'id' => $article_id ] );
		
		if ( $updated === false ) {
			// Log the actual database error for debugging
			error_log( 'AICS: Error al actualizar el artículo ' . $article_id . ' en la base de datos. MySQL Error: ' . $wpdb->last_error . ' Query: ' . $wpdb->last_query );
			wp_send_json_error( [ 'message' => 'Error al actualizar la base de datos. Por favor, revisa los logs del servidor para más detalles.' ], 500 );
		}
		// Si $updated es 0, significa que no se afectaron filas (los datos eran idénticos), lo cual no es un error.
		// Si $updated es > 0, significa que se afectaron filas.
		
		wp_send_json_success( [ 'message' => 'Cambios guardados.' ] );
	}

	/**
	 * NUEVO: Manejador para la llamada AJAX que regenera el contenido de un artículo.
	 */
	public function ajax_regenerate_content() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		// 2. Validar ID de artículo
		$article_id = isset( $_POST['article_id'] ) ? absint( $_POST['article_id'] ) : 0;
		if ( empty( $article_id ) ) {
			wp_send_json_error( [ 'message' => 'Falta el ID del artículo.' ], 400 );
		}

		global $wpdb;
		$table_name = $wpdb->prefix . 'aics_synthesized_articles';

		// 3. Obtener las URLs fuente del artículo original
		$source_urls_json = $wpdb->get_var( $wpdb->prepare( "SELECT source_urls FROM $table_name WHERE id = %d", $article_id ) );
		if ( ! $source_urls_json ) {
			wp_send_json_error( [ 'message' => 'No se encontraron las fuentes originales para este artículo.' ], 404 );
		}

		$source_urls = json_decode( $source_urls_json );
		if ( empty( $source_urls ) ) {
			wp_send_json_error( [ 'message' => 'Las fuentes originales están vacías.' ], 404 );
		}

		// 4. Re-construir el objeto de artículo(s) fuente para la IA
		// Esta es una simulación. Obtenemos el contenido de nuevo desde el feed.
		include_once( ABSPATH . WPINC . '/feed.php' );
		$article_group = [];
		foreach ( $source_urls as $url ) {
			// Este es un enfoque simplificado. No podemos obtener el contenido exacto del feed original
			// si ya no está, pero podemos usar la URL para crear un objeto básico.
			// La IA trabajará principalmente con el título si el contenido no se puede obtener.
			// Para una mejora futura, se podría guardar el contenido original en la BD.
			$article_group[] = (object) [
				'title'   => 'Artículo de ' . parse_url($url, PHP_URL_HOST), // Título genérico
				'content' => 'Contenido relacionado a la noticia encontrada en ' . $url, // Contenido genérico
				'url'     => $url
			];
		}

		// 5. Llamar al motor de la IA para regenerar
		$engine = new AICS_Engine( $this->options );
		$api_handler = $engine->_get_api_handler(); // Usamos el método público para obtener el handler
		$new_data = $api_handler->synthesize_articles( $article_group );

		if ( ! $new_data || ! isset( $new_data['titulo_sugerido'] ) ) {
			wp_send_json_error( [ 'message' => 'La IA no pudo regenerar el contenido. Revisa los logs para más detalles.' ], 500 );
		}

		// 6. Actualizar el artículo en la base de datos con el nuevo contenido
		$wpdb->update( $table_name, [
			'title' => sanitize_text_field( $new_data['titulo_sugerido'] ),
			'content' => wp_kses_post( $new_data['cuerpo_del_articulo'] ),
			'meta_description' => sanitize_text_field( $new_data['meta_descripcion_seo'] ),
			'suggested_tags' => isset( $new_data['sugerencia_etiquetas'] ) ? sanitize_text_field( $new_data['sugerencia_etiquetas'] ) : ''
		], [ 'id' => $article_id ] );

		// Preparamos la URL para "Crear Borrador" para devolverla en la respuesta AJAX
		$create_draft_url = wp_nonce_url(add_query_arg(['action' => 'create_draft', 'article_id' => $article_id]), 'aics_single_action_dashboard_' . $article_id);

		// Registrar el evento de éxito
		$log_message = sprintf('Contenido generado/regenerado para el artículo ID: %d. Título: "%s"', $article_id, sanitize_text_field($new_data['titulo_sugerido']));
		AICS_Logger::log('SUCCESS', $log_message, null, $article_id);

		// 7. Devolver los nuevos datos al frontend
		wp_send_json_success( [
			'title'   => $new_data['titulo_sugerido'],
			'content' => $new_data['cuerpo_del_articulo'],
			'tags'    => $new_data['sugerencia_etiquetas'] ?? '',
			'create_draft_url' => esc_url($create_draft_url) // Devolvemos la URL para la acción "Crear Borrador"
		] );
	}

	/**
	 * NUEVO: Manejador AJAX para cargar los feeds RSS manualmente.
	 */
	public function ajax_load_rss_feeds() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		$new_articles_count = $this->_load_rss_feeds_logic();

		// Registrar el evento
		$log_message = sprintf('Carga manual desde el dashboard. Se encontraron %d nuevos artículos.', $new_articles_count);
		AICS_Logger::log('INFO', $log_message);

		$message = sprintf( _n( '%d nuevo artículo ha sido cargado.', '%d nuevos artículos han sido cargados.', $new_articles_count, 'ai-content-synthesizer' ), $new_articles_count );
		wp_send_json_success( [ 'message' => $message ] );
	}

	/**
	 * NUEVO: Manejador AJAX para el botón "Forzar Ejecución Ahora" de la página de ajustes.
	 */
	public function ajax_force_load_feeds() {
		// 1. Verificación de seguridad
		if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], 'aics_ajax_nonce' ) ) {
			wp_send_json_error( [ 'message' => 'La verificación de seguridad ha fallado.' ], 403 );
		}

		$new_articles_count = $this->_load_rss_feeds_logic();

		// Registrar el evento
		$log_message = sprintf('Ejecución forzada desde ajustes. Se encontraron %d nuevos artículos.', $new_articles_count);
		AICS_Logger::log('INFO', $log_message);

		$message = sprintf( _n( '%d nuevo artículo ha sido cargado en el dashboard.', '%d nuevos artículos han sido cargados en el dashboard.', $new_articles_count, 'ai-content-synthesizer' ), $new_articles_count );
		wp_send_json_success( [ 'message' => $message ] );
	}

	/**
	 * Lógica interna para cargar los feeds RSS. Puede ser llamada por AJAX o por el cron.
	 *
	 * @return int El número de nuevos artículos cargados.
	 */
	private function _load_rss_feeds_logic() {
		global $wpdb;
		$table_name = $wpdb->prefix . 'aics_synthesized_articles';
		$options = get_option('aics_settings');
		$feeds = $options['rss_feeds'] ?? [];
		$new_articles_count = 0;

		if ( empty( $feeds ) ) {
			return 0; // No hay feeds, no se hace nada.
		}

		include_once( ABSPATH . WPINC . '/feed.php' );

		// Obtener todas las URLs de origen que ya existen en la BD para evitar duplicados.
		$existing_urls_results = $wpdb->get_col( "SELECT source_urls FROM $table_name" );
		$existing_urls = [];
		foreach ( $existing_urls_results as $json_urls ) {
			$urls = json_decode( $json_urls );
			if ( is_array( $urls ) ) {
				$existing_urls = array_merge( $existing_urls, $urls );
			}
		}
		$existing_urls = array_unique( $existing_urls );

		foreach ( $feeds as $feed_data ) {
			// Forzar la actualización del feed ignorando la caché de WordPress.
			add_filter( 'wp_feed_cache_transient_lifetime', '__return_zero' );
			$feed = fetch_feed( $feed_data['url'] );

			if ( ! is_wp_error( $feed ) ) {
				// Forzar una nueva descarga y reconstrucción del feed, ignorando la caché interna de SimplePie.
				// Esta es la solución definitiva para asegurar que siempre se obtienen los últimos artículos.
				$feed->force_feed(true);
				$feed->init(); // Volver a inicializar el feed para aplicar el force_feed.

				$maxitems = $feed->get_item_quantity( 10 ); // Cargar hasta 10 por feed
				$items = $feed->get_items( 0, $maxitems );

				foreach ( $items as $item ) {
					$item_url = $item->get_permalink();
					if ( ! in_array( $item_url, $existing_urls ) ) {
						$wpdb->insert( $table_name, [
							'created_at' => current_time( 'mysql' ),
							'title' => $item->get_title(),
							'content' => $item->get_description(), // Guardamos la descripción por si es útil
							'source_urls' => json_encode( [ $item_url ] ),
							'status' => 'pending_review'
						] );
						$new_articles_count++;
						$existing_urls[] = $item_url; // Añadir a la lista para evitar duplicados en la misma ejecución
					}
				}
			}
			// Restaurar el comportamiento normal de la caché.
			remove_filter( 'wp_feed_cache_transient_lifetime', '__return_zero' );
		}

		return $new_articles_count;
	}
}