<?php
/**
 * Contiene toda la lógica principal del proceso de curación y síntesis.
 */
class AICS_Engine {

    private $options;
    private $wpdb;

    public function __construct($options) {
        global $wpdb;
        $this->wpdb = $wpdb;
        $this->options = $options;
    }

    public function _get_api_handler() { // This was already public, but let's ensure it is.
        $ai_provider = $this->options['ai_provider'] ?? 'gemini';
        
        switch ($ai_provider) {
            case 'openai':
                if (!empty($this->options['openai_api_key'])) {
                    return new AICS_OpenAI_API($this->options);
                }
                break;
            case 'gemini':
            default:
                if (!empty($this->options['gemini_api_key'])) {
                    return new AICS_Gemini_API($this->options);
                }
                break;
        }
        return null;
    }
    
    /**
     * Obtiene una lista de todas las URLs de artículos que ya han sido procesadas
     * y utilizadas para generar contenido, para evitar duplicados.
     *
     * @return array Lista de URLs procesadas.
     */
    private function _get_all_processed_urls() {
        $table_name = $this->wpdb->prefix . 'aics_synthesized_articles';
        $results = $this->wpdb->get_col("SELECT source_urls FROM $table_name");
        
        $processed_urls = [];
        foreach ($results as $json_urls) {
            $urls = json_decode($json_urls, true);
            if (is_array($urls)) {
                $processed_urls = array_merge($processed_urls, $urls);
            }
        }
        
        return array_unique($processed_urls);
    }

    private function _fetch_all_new_articles() {
        include_once(ABSPATH . WPINC . '/feed.php');

        // Obtener todas las URLs que ya han sido procesadas para evitar duplicados.
        $processed_urls = $this->_get_all_processed_urls();

        $feeds = $this->options['rss_feeds'] ?? [];
        $enable_full_content_scraping = isset($this->options['enable_full_content_scraping']) && $this->options['enable_full_content_scraping'] == 1;
        $all_items = [];

        foreach ($feeds as $feed_data) {
            $feed_url = $feed_data['url'];
            $feed = fetch_feed($feed_url);
            if (!is_wp_error($feed)) {
                $maxitems = $feed->get_item_quantity(10);
                $items = $feed->get_items(0, $maxitems);
                
                foreach ($items as $item) {
                    $item_url = $item->get_permalink();

                    // Si la URL del artículo ya ha sido procesada, la ignoramos.
                    if (in_array($item_url, $processed_urls)) {
                        continue;
                    }

                    $article_content = $item->get_description(); // Contenido por defecto (del RSS)

                    // Si el scraping de contenido completo está activado, intentar obtenerlo.
                    if ($enable_full_content_scraping && !empty($item_url)) {
                        $scraped_content = $this->_scrape_full_content($item_url);
                        if ($scraped_content) {
                            $article_content = $scraped_content;
                            AICS_Logger::log('INFO', 'Contenido completo extraído de: ' . esc_url($item_url));
                        } else {
                            AICS_Logger::log('WARNING', 'Fallo al extraer contenido completo de: ' . esc_url($item_url) . '. Usando descripción del RSS.');
                        }
                    }

                    $all_items[] = (object) [
                        'title'   => $item->get_title(),
                        'content' => $article_content, // Puede ser el RSS o el scrapeado
                        'url'     => $item_url
                    ];
                }
            }
        }
        return $all_items;
    }

    /**
     * Intenta extraer el contenido principal de una URL.
     * Este es un método básico y puede no funcionar para todos los sitios.
     *
     * @param string $url La URL del artículo.
     * @return string|false El contenido extraído o false si falla.
     */
    private function _scrape_full_content($url) {
        $response = wp_remote_get($url, ['timeout' => 30]); // Aumentar timeout para scraping

        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) !== 200) {
            return false;
        }

        $html = wp_remote_retrieve_body($response);
        
        // Usar una librería como Readability.php o un parser más avanzado sería ideal.
        // Para este ejemplo, intentaremos una extracción básica de elementos comunes de contenido.
        // Esto es muy básico y puede fallar en muchos sitios.
        preg_match('/<article[^>]*>(.*?)<\/article>/is', $html, $matches);
        if (isset($matches[1]) && !empty(strip_tags($matches[1]))) {
            return $matches[1];
        }
        preg_match('/<div[^>]*class=["\'][^"\']*entry-content[^"\']*["\'][^>]*>(.*?)<\/div>/is', $html, $matches);
        if (isset($matches[1]) && !empty(strip_tags($matches[1]))) {
            return $matches[1];
        }
        preg_match('/<div[^>]*class=["\'][^"\']*post-content[^"\']*["\'][^>]*>(.*?)<\/div>/is', $html, $matches);
        if (isset($matches[1]) && !empty(strip_tags($matches[1]))) {
            return $matches[1];
        }
        preg_match('/<main[^>]*>(.*?)<\/main>/is', $html, $matches);
        if (isset($matches[1]) && !empty(strip_tags($matches[1]))) {
            return $matches[1];
        }

        return false; // No se pudo extraer el contenido principal
    }

    private function _save_synthesized_article($data, $cluster) {
        $table_name = $this->wpdb->prefix . 'aics_synthesized_articles';
        
        $source_urls = [];
        foreach($cluster as $article) { $source_urls[] = $article->url; }

        $inserted = $this->wpdb->insert($table_name, [
            'created_at'        => current_time('mysql'),
            'title'             => sanitize_text_field($data['titulo_sugerido']),
            'content'           => wp_kses_post($data['cuerpo_del_articulo']),
            'meta_description'  => sanitize_text_field($data['meta_descripcion_seo']),
            'suggested_tags'    => isset($data['sugerencia_etiquetas']) ? sanitize_text_field($data['sugerencia_etiquetas']) : null,
            'source_urls'       => json_encode($source_urls),
            'status'            => 'pending_review'
        ]);

        return $inserted ? $this->wpdb->insert_id : false;
    }
}