<?php
class AICS_Gemini_API {

    use AICS_Api_Style_Helper;

    private $api_key;
    private $writing_style; // NUEVO
    private $article_length; // NUEVO
    private $max_prompt_chars; // NUEVO
    private $api_url;

    public function __construct($options) {
        $this->api_key = $options['gemini_api_key'] ?? '';
        $this->writing_style = $options['writing_style'] ?? 'formal'; // NUEVO
        $this->article_length = $options['article_length'] ?? 'medium'; // NUEVO
        $this->max_prompt_chars = $options['max_prompt_chars'] ?? 3000; // NUEVO

        // Permitir seleccionar el modelo desde las opciones
        $model = $options['gemini_model'] ?? 'gemini-2.5-flash';
        $this->api_url = "https://generativelanguage.googleapis.com/v1beta/models/{$model}:generateContent?key=";
    }

    public function synthesize_articles($articles_to_synthesize) {
        $prompt = $this->build_prompt($articles_to_synthesize);
        // ... (el resto de la función se mantiene igual)
        $request_body = [ 'contents' => [ ['parts' => [ ['text' => $prompt] ] ] ], 'generationConfig' => [ 'response_mime_type' => 'application/json', ] ];
        $response = wp_remote_post($this->api_url . $this->api_key, [ 'method' => 'POST', 'headers' => ['Content-Type' => 'application/json'], 'body' => json_encode($request_body), 'timeout' => 120, ]);
        
        if (is_wp_error($response)) { 
            AICS_Logger::log('ERROR', 'Error de WP en la llamada a la API de Gemini: ' . $response->get_error_message());
            return false; 
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body);

        // MEJORA: Capturar errores específicos devueltos por la API de Gemini
        if (isset($data->error)) {
            $error_message = 'Error de la API de Gemini: ' . ($data->error->message ?? 'Mensaje no disponible.');
            AICS_Logger::log('ERROR', $error_message, json_decode($response_body, true));
            return false;
        }

        $gemini_json_string = $data->candidates[0]->content->parts[0]->text ?? '';
        if (empty($gemini_json_string)) { return false; }
        $synthesized_data = json_decode(trim($gemini_json_string), true);
        if (json_last_error() === JSON_ERROR_NONE && isset($synthesized_data['titulo_sugerido'])) { return $synthesized_data; }
        return false;
    }

    public function score_article_interest($article, $interest_topics) {
        $excerpt = mb_substr(strip_tags($article->content), 0, 200);
        $prompt = "Eres un editor experto. Basado en el siguiente título y resumen, puntúa el interés periodístico del 1 al 10 en relación a estos temas: '{$interest_topics}'. Responde solo con el número (ej: 8). Título: '{$article->title}'. Resumen: '{$excerpt}'";

        $request_body = [
            'contents' => [
                ['parts' => [ ['text' => $prompt] ] ]
            ]
        ];

        $response = wp_remote_post($this->api_url . $this->api_key, [
            'method' => 'POST',
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode($request_body),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) { return 0; }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body);
        $score_text = $data->candidates[0]->content->parts[0]->text ?? '0';

        return intval(preg_replace('/[^0-9]/', '', $score_text));
    }

    private function get_length_instruction() {
        switch ($this->article_length) {
            case 'short':
                return 'de aproximadamente 200 palabras';
            case 'long':
                return 'de al menos 600 palabras';
            case 'medium':
            default:
                return 'de al menos 400 palabras';
        }
    }

    private function build_prompt($articles) {
        $style_instruction = $this->get_style_instruction(); // NUEVO
        $length_instruction = $this->get_length_instruction(); // NUEVO

        $prompt = "Tu rol es ser un periodista experto en redacción y análisis. A partir de los siguientes fragmentos de noticias, redacta un artículo de noticias profundo, detallado y bien estructurado $length_instruction. Para mejorar la legibilidad, divide el artículo en secciones lógicas utilizando subtítulos (etiquetas `<h2>`). Elabora sobre los puntos clave, añade contexto si es posible y asegúrate de que el resultado sea coherente y original. No copies frases directamente. $style_instruction El artículo debe estar en español.\n\n";
        $prompt .= "Fuentes a sintetizar:\n";
        
        $i = 1;
        foreach ($articles as $article) {
            // MEJORA: Limpieza avanzada del contenido antes de enviarlo a la IA.
            $clean_content = strip_tags($article->content); // 1. Quitar HTML
            $clean_content = html_entity_decode($clean_content); // 2. Decodificar entidades como &nbsp;
            $clean_content = preg_replace('/\s+/', ' ', $clean_content); // 3. Reemplazar múltiples espacios/saltos por uno solo
            $excerpt = mb_substr($clean_content, 0, $this->max_prompt_chars); // 4. Usamos el ajuste de caracteres
            $prompt .= "Noticia " . $i++ . ": Título: '" . $article->title . "'. Resumen: '" . $excerpt . " ...'\n";
        }

        $prompt .= "\nLa salida debe ser estrictamente un objeto JSON con las siguientes claves: \"titulo_sugerido\", \"cuerpo_del_articulo\" (en formato HTML, usando párrafos `<p>` y subtítulos `<h2>` para estructurar el contenido), \"meta_descripcion_seo\", \"sugerencia_etiquetas\" (una cadena de texto con 5 a 7 etiquetas SEO relevantes, separadas por comas).";

        return $prompt;
    }
}