<?php
/**
 * Maneja toda la comunicación con la API de OpenAI (ChatGPT).
 */

class AICS_OpenAI_API {

    use AICS_Api_Style_Helper;

    private $api_key;
    private $writing_style; // NUEVO
    private $article_length; // NUEVO
    private $max_prompt_chars; // NUEVO
    private $api_url = 'https://api.openai.com/v1/chat/completions';

    public function __construct($options) {
        $this->api_key = $options['openai_api_key'] ?? '';
        $this->writing_style = $options['writing_style'] ?? 'formal'; // NUEVO
        $this->article_length = $options['article_length'] ?? 'medium'; // NUEVO
        $this->max_prompt_chars = $options['max_prompt_chars'] ?? 3000; // NUEVO
    }

    public function synthesize_articles($articles_to_synthesize) {
        $messages = $this->build_messages($articles_to_synthesize);
        // ... (el resto de la función se mantiene igual)
        $request_body = [ 'model' => 'gpt-3.5-turbo', 'messages' => $messages, 'response_format' => ['type' => 'json_object'] ];
        $response = wp_remote_post($this->api_url, [ 'method' => 'POST', 'headers' => [ 'Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $this->api_key, ], 'body' => json_encode($request_body), 'timeout' => 120, ]);
        
        if (is_wp_error($response)) { 
            AICS_Logger::log('ERROR', 'Error de WP en la llamada a la API de OpenAI: ' . $response->get_error_message());
            return false; 
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body);

        // MEJORA: Capturar errores específicos devueltos por la API de OpenAI
        if (isset($data->error)) {
            $error_message = 'Error de la API de OpenAI: ' . ($data->error->message ?? 'Mensaje no disponible.');
            AICS_Logger::log('ERROR', $error_message, json_decode($response_body, true));
            return false;
        }

        $openai_json_string = $data->choices[0]->message->content ?? '';
        if (empty($openai_json_string)) { return false; }
        $synthesized_data = json_decode(trim($openai_json_string), true);
        if (json_last_error() === JSON_ERROR_NONE && isset($synthesized_data['titulo_sugerido'])) { return $synthesized_data; }
        return false;
    }

    public function score_article_interest($article, $interest_topics) {
        $excerpt = mb_substr(strip_tags($article->content), 0, 200);
        $system_prompt = "Eres un editor experto. El usuario te dará un título y un resumen. Debes puntuar su interés periodístico del 1 al 10 en relación a estos temas: '{$interest_topics}'. Responde solo con el número (ej: 8).";
        $user_content = "Título: '{$article->title}'. Resumen: '{$excerpt}'";

        $messages = [
            ['role' => 'system', 'content' => $system_prompt],
            ['role' => 'user', 'content' => $user_content]
        ];

        $request_body = [
            'model' => 'gpt-3.5-turbo',
            'messages' => $messages,
            'max_tokens' => 5,
        ];

        $response = wp_remote_post($this->api_url, [
            'method' => 'POST',
            'headers' => ['Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $this->api_key],
            'body' => json_encode($request_body),
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) { return 0; }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body);
        $score_text = $data->choices[0]->message->content ?? '0';

        return intval(preg_replace('/[^0-9]/', '', $score_text));
    }

    private function get_length_instruction() {
        switch ($this->article_length) {
            case 'short':
                return 'de aproximadamente 200 palabras';
            case 'long':
                return 'de al menos 600 palabras';
            case 'medium':
            default:
                return 'de al menos 400 palabras';
        }
    }

    private function build_messages($articles) {
        $style_instruction = $this->get_style_instruction(); // NUEVO
        $length_instruction = $this->get_length_instruction(); // NUEVO

        $system_prompt = "Tu rol es ser un periodista experto en redacción y análisis. A partir de los fragmentos de noticias que te proporcionará el usuario, debes redactar un artículo de noticias profundo, detallado y bien estructurado $length_instruction. Para mejorar la legibilidad, divide el artículo en secciones lógicas utilizando subtítulos (etiquetas `<h2>`). Elabora sobre los puntos clave, añade contexto si es posible y asegúrate de que el resultado sea coherente y original. No copies frases directamente. $style_instruction El artículo debe estar en español. Debes devolver estrictamente un objeto JSON con las claves: \"titulo_sugerido\", \"cuerpo_del_articulo\" (en formato HTML, usando párrafos `<p>` y subtítulos `<h2>` para estructurar el contenido), \"meta_descripcion_seo\", \"sugerencia_etiquetas\" (una cadena de texto con 5 a 7 etiquetas SEO relevantes, separadas por comas).";

        $user_content = "Fuentes a sintetizar:\n";
        $i = 1;
        foreach ($articles as $article) {
            // MEJORA: Limpieza avanzada del contenido antes de enviarlo a la IA.
            $clean_content = strip_tags($article->content); // 1. Quitar HTML
            $clean_content = html_entity_decode($clean_content); // 2. Decodificar entidades como &nbsp;
            $clean_content = preg_replace('/\s+/', ' ', $clean_content); // 3. Reemplazar múltiples espacios/saltos por uno solo
            $excerpt = mb_substr($clean_content, 0, $this->max_prompt_chars); // 4. Usamos el ajuste de caracteres
            $user_content .= "Noticia " . $i++ . ": Título: '" . $article->title . "'. Resumen: '" . $excerpt . " ...'\n";
        }

        return [
            ['role' => 'system', 'content' => $system_prompt],
            ['role' => 'user', 'content' => $user_content]
        ];
    }
}